#!/usr/bin/env bash
set -u

# ==============================================================================
# Usage:
#   ./analyze_query.sh <query_file_1> [query_file_2] ...
#
# Description:
#   Runs EXPLAIN (ANALYZE, VERBOSE, BUFFERS) for specific SQL queries across
#   defined algorithm configurations.
#   It automatically handles filenames with or without the ".sql" extension.
#
# Example:
#   ./analyze_query.sh 10a
#   ./analyze_query.sh 10a.sql 17b.sql
# ==============================================================================

# ===== Configuration =====
DB_NAME="${DB_NAME:-job_imdb}"
RESULT_DIR="${RESULT_DIR:-results}"
LOG_FILE="${RESULT_DIR}/analysis_run.log"

# Ensure output directory exists
mkdir -p "$RESULT_DIR"
: > "$LOG_FILE"

# Algorithm Configurations
ALGOS=("dp" "goo_cost" "goo_result_size" "goo_combined" "geqo")

log() {
  local msg="$(date '+%Y-%m-%d %H:%M:%S') $@"
  echo "$msg"
  echo "$msg" >> "$LOG_FILE"
}

get_guc_sql() {
  local algo="$1"
  case "$algo" in
    dp)
      echo "SET geqo_threshold = 100; SET enable_goo_join_search = off;" ;;
    goo_cost)
      echo "SET geqo_threshold = 2; SET enable_goo_join_search = on; SET goo_greedy_strategy = 'cost';" ;;
    goo_result_size)
      echo "SET geqo_threshold = 2; SET enable_goo_join_search = on; SET goo_greedy_strategy = 'result_size';" ;;
    goo_combined)
      echo "SET geqo_threshold = 2; SET enable_goo_join_search = on; SET goo_greedy_strategy = 'combined';" ;;
    geqo)
      echo "SET geqo_threshold = 2; SET enable_goo_join_search = off;" ;;
    *)
      log "Error: Unknown algo: $algo"; return 1 
      ;;
  esac
}

# ===== Input Validation =====
if [ "$#" -eq 0 ]; then
    echo "Error: No input files provided."
    echo "Usage: $0 file1 [file2 ...]"
    exit 1
fi

log "========== Analysis Start =========="
log "DB Name:    $DB_NAME"
log "Result Dir: $RESULT_DIR"

# ===== Main Loop =====
for input_arg in "$@"; do
    
    # [Feature] Auto-append .sql suffix if missing
    if [[ "$input_arg" != *.sql ]]; then
        sql_file="${input_arg}.sql"
    else
        sql_file="$input_arg"
    fi
    
    # 1. Check if file exists
    if [ ! -f "$sql_file" ]; then
        log "[Error] File not found: $sql_file (Input was: $input_arg), skipping..."
        continue
    fi

    base_name=$(basename "$sql_file" .sql)
    report_file="${RESULT_DIR}/${base_name}_plan.txt"
    
    # Initialize Report File
    {
        echo "========================================================"
        echo " Detailed Analysis for Query: $base_name "
        echo " Source File: $sql_file"
        echo " Date: $(date)"
        echo "========================================================"
        echo ""
    } > "$report_file"

    log "Analyzing: $sql_file -> $report_file"

    # 2. Iterate through Algo Configs
    for algo in "${ALGOS[@]}"; do
        log "  -> Running Config: $algo"

        tmp_sql=$(mktemp)
        
        # A. Write Section Header
        {
            echo ""
            echo "################################################################"
            echo "# CONFIGURATION: $algo"
            echo "################################################################"
            echo ""
        } >> "$report_file"

        # B. Construct Execution SQL
        get_guc_sql "$algo" >> "$tmp_sql"
        echo "EXPLAIN (ANALYZE, VERBOSE, BUFFERS)" >> "$tmp_sql"
        cat "$sql_file" >> "$tmp_sql"

        # C. Execute and Append Results
        if psql -d "$DB_NAME" -f "$tmp_sql" >> "$report_file" 2>&1; then
             : # Success
        else
             log "     [Error] Execution failed for $algo"
             echo "ERROR: Execution failed. Check log." >> "$report_file"
        fi

        rm -f "$tmp_sql"
    done

    log "Finished: $base_name"
done

log "========== All Done =========="
log "Check folder '$RESULT_DIR' for results."